/*********************************************************************
 *
 *  Gmsh tutorial 3
 *
 *  Extruded meshes, parameters, options
 *
 *********************************************************************/

// Again, we start by including the first tutorial:

Include "t1.geo";

// As in `t2.geo', we plan to perform an extrusion along the z axis.
// But here, instead of only extruding the geometry, we also want to
// extrude the 2D mesh. This is done with the same `Extrude' command,
// but by specifying element 'Layers' (2 layers in this case, the
// first one with 8 subdivisions and the second one with 2
// subdivisions, both with a height of h/2):

h = 0.1;

Extrude {0,0,h} {
  Surface{6}; Layers{ {8,2}, {0.5,1} };
}

// The extrusion can also be performed with a rotation instead of a
// translation, and the resulting mesh can be recombined into prisms
// (we use only one layer here, with 7 subdivisions). All rotations
// are specified by an axis direction ({0,1,0}), an axis point
// ({-0.1,0,0.1}) and a rotation angle (-Pi/2):

Extrude { {0,1,0} , {-0.1,0,0.1} , -Pi/2 } {
  Surface{122}; Layers{7}; Recombine;
}

// Note that a translation ({-2*h,0,0}) and a rotation ({1,0,0},
// {0,0.15,0.25}, Pi/2) can also be combined. Here the angle is
// specified as a 'parameter', using the 'DefineConstant' syntax.
// This parameter can be modified insteractively in the GUI, and
// can be exchanged with other codes using the ONELAB framework:

DefineConstant[ angle = {90, Min 0, Max 120, Step 1,
                         Name "Parameters/Twisting angle"} ];

out[] = Extrude { {-2*h,0,0}, {1,0,0} , {0,0.15,0.25} , angle * Pi / 180 } {
  Surface{144}; Layers{10}; Recombine;
};

// In this last extrusion command we retrieved the volume number
// programatically by using the return value (a list) of the Extrude
// command. This list contains the "top" of the extruded surface (in
// out[0]), the newly created volume (in out[1]) and the ids of the
// lateral surfaces (in out[2], out[3], ...)

// We can then define a new physical volume to save all the tetrahedra
// with a common region number (101):

Physical Volume(101) = {1, 2, out[1]};

// Let us now change some options... Since all interactive options are
// accessible in Gmsh's scripting language, we can for example make point tags
// visible or redefine some colors directly in the input file:

Geometry.PointNumbers = 1;
Geometry.Color.Points = Orange;
General.Color.Text = White;
Mesh.Color.Points = {255,0,0};

// Note that all colors can be defined literally or numerically, i.e.
// `Mesh.Color.Points = Red' is equivalent to `Mesh.Color.Points =
// {255,0,0}'; and also note that, as with user-defined variables, the
// options can be used either as right or left hand sides, so that the
// following command will set the surface color to the same color as
// the points:

Geometry.Color.Surfaces = Geometry.Color.Points;

// You can use the `Help->Current options' menu to see the current
// values of all options. To save all the options in a file, use
// `File->Save as->Gmsh options'. To associate the current options
// with the current file use `File->Save Options->For Current
// File'. To save the current options for all future Gmsh sessions use
// `File->Save Options->As default'.
